<?php
/**
 * Custom Post Types module parameters
 *
 * @package WordPress
 * @subpackage Decibel
 * @version 3.3.2
 */

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'Wolf_Core' ) ) {
	return;
}

/**
 * Common parameters  usedacross post type modules
 */
function decibel_common_module_params( $animation_condition = array(), $post_count_condition = array() ) {
	return apply_filters(
		'decibel_common_module_params',
		array(
			array(
				'type'        => 'text',
				'label'       => esc_html__( 'Index ID', 'decibel' ),
				'default'     => 'index-' . wp_rand( 0, 99999 ),
				'param_name'  => 'el_id',
				'description' => esc_html__( 'A unique identifier for the post module (required).', 'decibel' ),
			),
			array(
				'label'       => esc_html__( 'Animation', 'decibel' ),
				'param_name'  => 'item_animation',
				'type'        => 'select',
				'options'     => decibel_get_animations(),
				'default'     => 'none',
				'admin_label' => true,
				'condition'   => $animation_condition,
			),

			array(
				'label'       => esc_html__( 'Number of Posts', 'decibel' ),
				'param_name'  => 'posts_per_page',
				'type'        => 'text',
				'placeholder' => 9,
				'description' => esc_html__( 'Leave empty to display all post at once.', 'decibel' ),
				'admin_label' => true,
				'condition'   => $post_count_condition,
			),

			array(
				'type'        => 'text',
				'label'       => esc_html__( 'Offset', 'decibel' ),
				'param_name'  => 'offset',
				'description' => esc_html__( 'The amount of posts that should be skipped in the beginning of the query. If an offset is set, sticky posts will be ignored.', 'decibel' ),
				'group'       => esc_html__( 'Query', 'decibel' ),
				'admin_label' => true,
			),

			array(
				'label'       => esc_html__( 'Quick CSS', 'decibel' ),
				'description' => esc_html__( 'CSS inline style', 'decibel' ),
				'param_name'  => 'inline_style',
				'type'        => 'textarea',
				'group'       => esc_html__( 'Extra', 'decibel' ),
			),
		)
	);
}

/**
 * Overlay appearance parameters used across post type modules
 *
 * @param string $post_type The post type, duh.
 */
function decibel_overlay_module_params( $post_type ) {
	return apply_filters(
		'decibel_overlay_module_params',
		array(
			/* Overlay Color for VC */
			array(
				'type'               => 'select',
				'label'              => esc_html__( 'Overlay Color', 'decibel' ),
				'param_name'         => 'overlay_color',
				'options'            => array_merge(
					array( 'auto' => esc_html__( 'Auto', 'decibel' ) ),
					decibel_shared_gradient_colors(),
					decibel_shared_colors(),
					array( 'custom' => esc_html__( 'Custom color', 'decibel' ) )
				),
				'default'            => apply_filters( 'wolf_core_default_item_overlay_color', 'black' ),
				'description'        => esc_html__( 'Select an overlay color.', 'decibel' ),
				'param_holder_class' => 'wolf_core_colored-select',
				'condition'          => array(
					$post_type . '_layout' => array( 'overlay' ),
				),
				'page_builder'       => 'vc',
			),

			array(
				'type'         => 'colorpicker',
				'label'        => esc_html__( 'Overlay Custom Color', 'decibel' ),
				'param_name'   => 'overlay_custom_color',
				'condition'    => array(
					$post_type . '_layout' => array( 'overlay' ),
					'overlay_color'        => array( 'custom' ),
				),
				'page_builder' => 'vc',
			),

			/* Overlay Color for Elementor */
			array(
				'label'        => esc_html__( 'Overlay Color', 'decibel' ),
				'type'         => 'select',
				'options'      => array_merge(
					array( 'auto' => esc_html__( 'Auto', 'decibel' ) ),
					decibel_shared_colors(),
					array( 'custom' => esc_html__( 'Custom color', 'decibel' ) )
				),
				'param_name'   => 'overlay_color',
				'default'      => 'auto',
				'page_builder' => 'elementor',
				'condition'    => array(
					$post_type . '_layout' => array( 'overlay' ),
				),
				'group'        => esc_html__( 'Style', 'decibel' ),
			),

			array(
				'type'         => 'colorpicker',
				'label'        => esc_html__( 'Overlay Color', 'decibel' ),
				'param_name'   => 'overlay_custom_color',
				'page_builder' => 'elementor',
				'selectors'    => array(
					'{{WRAPPER}} .bg-overlay' => 'background-color: {{VALUE}}!important;',
				),
				'group'        => esc_html__( 'Style', 'decibel' ),
				'condition'    => array(
					$post_type . '_layout' => array( 'overlay' ),
					'overlay_color'        => array( 'custom' ),
				),
				'page_builder' => 'elementor',
			),

			/* Overlay Opacity */
			array(
				'type'         => 'slider',
				'label'        => esc_html__( 'Overlay Opacity', 'decibel' ),
				'param_name'   => 'overlay_opacity',
				'min'          => 0,
				'max'          => 1,
				'step'         => 0.01,
				'default'      => apply_filters( 'wolf_core_default_item_overlay_opacity', 40 ) / 100,
				'condition'    => array(
					$post_type . '_layout' => array( 'overlay' ),
				),
				'selectors'    => array(
					'{{WRAPPER}} .bg-overlay' => 'opacity: {{SIZE}}!important;',
				),
				'group'        => esc_html__( 'Style', 'decibel' ),
				'page_builder' => 'elementor',
			),

			array(
				'type'         => 'slider',
				'label'        => esc_html__( 'Overlay Opacity in Percent', 'decibel' ),
				'param_name'   => 'overlay_opacity',
				'min'          => 0,
				'max'          => 100,
				'step'         => 1,
				'default'      => apply_filters( 'wolf_core_default_item_overlay_opacity', 40 ),
				'condition'    => array(
					$post_type . '_layout' => array( 'overlay' ),
				),
				'group'        => esc_html__( 'Style', 'decibel' ),
				'page_builder' => 'vc',
			),

			/* Overlay Text Color for VC */
			array(
				'type'               => 'select',
				'label'              => esc_html__( 'Overlay Text Color', 'decibel' ),
				'param_name'         => 'overlay_text_color',
				'options'            => array_merge(
					array( 'auto' => esc_html__( 'Auto', 'decibel' ) ),
					decibel_shared_gradient_colors(),
					decibel_shared_colors(),
					array( 'custom' => esc_html__( 'Custom color', 'decibel' ) )
				),
				'default'            => apply_filters( 'wolf_core_default_item_overlay_color', 'black' ),
				'description'        => esc_html__( 'Select an overlay color.', 'decibel' ),
				'param_holder_class' => 'wolf_core_colored-select',
				'condition'          => array(
					$post_type . '_layout' => array( 'overlay' ),
				),
				'page_builder'       => 'vc',
			),

			array(
				'type'         => 'colorpicker',
				'label'        => esc_html__( 'Overlay Custom Text Color', 'decibel' ),
				'param_name'   => 'overlay_text_custom_color',
				'condition'    => array(
					$post_type . '_layout' => array( 'overlay' ),
					'overlay_text_color'   => array( 'custom' ),
				),
				'page_builder' => 'vc',
			),

			/* Overlay Text Color for Elementor */
			array(
				'label'        => esc_html__( 'Overlay Text Color', 'decibel' ),
				'type'         => 'hidden',
				'param_name'   => 'overlay_text_color',
				'default'      => 'custom',
				'condition'    => array(
					$post_type . '_layout' => array( 'overlay' ),
				),
				'page_builder' => 'elementor',
			),

			array(
				'type'         => 'colorpicker',
				'label'        => esc_html__( 'Overlay Text Color', 'decibel' ),
				'param_name'   => 'overlay_text_custom_color',
				'page_builder' => 'elementor',
				'selectors'    => array(
					'{{WRAPPER}} .entry-summary' => 'color: {{VALUE}}!important;',
				),
				'condition'    => array(
					$post_type . '_layout' => array( 'overlay' ),
				),
				'group'        => esc_html__( 'Style', 'decibel' ),
				'page_builder' => 'elementor',
			),
		)
	);
}

/**
 * Post Index
 */
function decibel_post_index_params() {
	return apply_filters(
		'decibel_post_index_params',
		array(
			'properties' => array(
				'name'          => esc_html__( 'Posts', 'decibel' ),
				'description'   => esc_html__( 'Display your posts using the theme layouts', 'decibel' ),
				'vc_base'       => 'wolf_core_post_index',
				'el_base'       => 'post-index',
				'vc_category'   => esc_html__( 'Content', 'decibel' ),
				'el_categories' => array( 'post-modules' ),
				'icon'          => 'linea-arrows linea-arrows-squares',
				'weight'        => 999,
			),

			'params'     => array_merge(
				decibel_common_module_params(),
				array(
					array(
						'param_name'  => 'post_display',
						'label'       => esc_html__( 'Post Display', 'decibel' ),
						'type'        => 'select',
						'options'     => apply_filters(
							'decibel_post_display_options',
							array(
								'standard' => esc_html__( 'Standard', 'decibel' ),
								'grid'     => esc_html__( 'Grid', 'decibel' ),
							)
						),
						'default'     => 'grid',
						'admin_label' => true,
					),

					array(
						'param_name'  => 'post_metro_pattern',
						'label'       => esc_html__( 'Metro Pattern', 'decibel' ),
						'type'        => 'select',
						'options'     => decibel_get_metro_patterns(),
						'default'     => 'auto',
						'condition'   => array(
							'post_display' => array( 'metro_modern_alt', 'metro' ),
						),
						'admin_label' => true,
					),

					array(
						'param_name'   => 'post_alternate_thumbnail_position',
						'label'        => esc_html__( 'Alternate thumbnail position', 'decibel' ),
						'type'         => 'checkbox',
						'label_on'     => esc_html__( 'Yes', 'decibel' ),
						'label_off'    => esc_html__( 'No', 'decibel' ),
						'return_value' => 'no',
						'condition'    => array(
							'post_display' => array( 'lateral' ),
						),
					),

					array(
						'param_name'  => 'post_module',
						'label'       => esc_html__( 'Module', 'decibel' ),
						'type'        => 'select',
						'options'     => array(
							'grid'     => esc_html__( 'Grid', 'decibel' ),
							'carousel' => esc_html__( 'Carousel', 'decibel' ),
						),
						'description' => esc_html__( 'The carousel is not visible in preview mode yet.', 'decibel' ),
						'default'     => 'grid',
						'admin_label' => true,
						'condition'   => array(
							'post_display' => array( 'grid', 'grid_classic', 'grid_modern' ),
						),
					),

					array(
						'param_name' => 'post_excerpt_length',
						'label'      => esc_html__( 'Post Excerpt Lenght', 'decibel' ),
						'type'       => 'select',
						'options'    => array(
							'shorten' => esc_html__( 'Shorten', 'decibel' ),
							'full'    => esc_html__( 'Full', 'decibel' ),
						),
						'default'    => 'shorten',
						'condition'  => array(
							'post_display' => array( 'masonry' ),
						),
					),

					array(
						'param_name'   => 'post_display_elements',
						'label'        => esc_html__( 'Elements', 'decibel' ),
						'type'         => 'group_checkbox',
						'options'      => array(
							'show_thumbnail'  => esc_html__( 'Thumbnail', 'decibel' ),
							'show_date'       => esc_html__( 'Date', 'decibel' ),
							'show_text'       => esc_html__( 'Text', 'decibel' ),
							'show_category'   => esc_html__( 'Category', 'decibel' ),
							'show_author'     => esc_html__( 'Author', 'decibel' ),
							'show_tags'       => esc_html__( 'Tags', 'decibel' ),
							'show_extra_meta' => esc_html__( 'Extra Meta', 'decibel' ),
						),
						'default'      => 'show_thumbnail,show_date,show_text,show_author,show_category',
						'description'  => esc_html__( 'Note that some options may be ignored depending on the post display.', 'decibel' ),
						'admin_label'  => true,
						'page_builder' => 'vc',
					),

					array(
						'param_name'   => 'post_show_thumbnail',
						'label'        => esc_html__( 'Show Thumbnail', 'decibel' ),
						'type'         => 'checkbox',
						'default'      => 'yes',
						'page_builder' => 'elementor',
					),

					array(
						'param_name'   => 'post_show_date',
						'label'        => esc_html__( 'Show Date', 'decibel' ),
						'type'         => 'checkbox',
						'default'      => 'yes',
						'page_builder' => 'elementor',
					),

					array(
						'param_name'   => 'post_show_text',
						'label'        => esc_html__( 'Show Text', 'decibel' ),
						'type'         => 'checkbox',
						'default'      => 'yes',
						'page_builder' => 'elementor',
					),

					array(
						'param_name'   => 'post_show_category',
						'label'        => esc_html__( 'Show Category', 'decibel' ),
						'type'         => 'checkbox',
						'default'      => 'yes',
						'page_builder' => 'elementor',
					),

					array(
						'param_name'   => 'post_show_author',
						'label'        => esc_html__( 'Show Author', 'decibel' ),
						'type'         => 'checkbox',
						'default'      => 'yes',
						'page_builder' => 'elementor',
					),

					array(
						'param_name'   => 'post_show_tags',
						'label'        => esc_html__( 'Show Tags', 'decibel' ),
						'type'         => 'checkbox',
						'default'      => 'yes',
						'page_builder' => 'elementor',
					),

					array(
						'param_name'  => 'post_excerpt_type',
						'label'       => esc_html__( 'Post Excerpt Type', 'decibel' ),
						'type'        => 'select',
						'options'     => array(
							'auto'   => esc_html__( 'Auto', 'decibel' ),
							'manual' => esc_html__( 'Manual', 'decibel' ),
						),
						'default'     => 'auto',
						'description' => sprintf(
							decibel_kses( __( 'When using the manual excerpt, you must split your post using a "More Tag".', 'decibel' ) ),
							esc_url( 'https://en.support.wordpress.com/more-tag/' )
						),
						'condition'   => array(
							'post_display' => array( 'standard', 'standard_modern' ),
						),
					),

					array(
						'param_name'  => 'grid_padding',
						'label'       => esc_html__( 'Padding', 'decibel' ),
						'type'        => 'select',
						'options'     => array(
							'yes' => esc_html__( 'Yes', 'decibel' ),
							'no'  => esc_html__( 'No', 'decibel' ),
						),
						'default'     => 'yes',
						'admin_label' => true,
						'condition'   => array(
							'post_display' => array( 'grid', 'masonry', 'metro' ),
						),
					),

					array(
						'param_name'  => 'pagination',
						'label'       => esc_html__( 'Pagination', 'decibel' ),
						'type'        => 'select',
						'options'     => array(
							'none'                => esc_html__( 'None', 'decibel' ),
							'load_more'           => esc_html__( 'Load More', 'decibel' ),
							'standard_pagination' => esc_html__( 'Numeric Pagination', 'decibel' ),
							'link_to_blog'        => esc_html__( 'Link to Blog Archives', 'decibel' ),
						),
						'default'     => 'none',
						'admin_label' => true,
					),

					array(
						'type'         => 'checkbox',
						'label'        => esc_html__( 'Ignore Sticky Posts', 'decibel' ),
						'param_name'   => 'ignore_sticky_posts',
						'label_on'     => esc_html__( 'Yes', 'decibel' ),
						'label_off'    => esc_html__( 'No', 'decibel' ),
						'return_value' => 'yes',
						'description'  => esc_html__( 'It will still include the sticky posts but it will not prioritize them in the query.', 'decibel' ),
						'group'        => esc_html__( 'Query', 'decibel' ),
					),

					array(
						'type'         => 'checkbox',
						'label'        => esc_html__( 'Exclude Sticky Posts', 'decibel' ),
						'description'  => esc_html__( 'It will still exclude the sticky posts.', 'decibel' ),
						'param_name'   => 'exclude_sticky_posts',
						'label_on'     => esc_html__( 'Yes', 'decibel' ),
						'label_off'    => esc_html__( 'No', 'decibel' ),
						'return_value' => 'yes',
						'group'        => esc_html__( 'Query', 'decibel' ),
					),

					array(
						'type'        => 'text',
						'label'       => esc_html__( 'Category', 'decibel' ),
						'param_name'  => 'category',
						'description' => esc_html__( 'Include only one or several categories. Paste category slug(s) separated by a comma', 'decibel' ),
						'placeholder' => esc_html__( 'my-category, other-category', 'decibel' ),
						'group'       => esc_html__( 'Query', 'decibel' ),
					),

					array(
						'type'        => 'text',
						'label'       => esc_html__( 'Exclude Category by ID', 'decibel' ),
						'param_name'  => 'category_exclude',
						'description' => esc_html__( 'Exclude only one or several categories. Paste category ID(s) separated by a comma', 'decibel' ),
						'placeholder' => esc_html__( '456, 756', 'decibel' ),
						'group'       => esc_html__( 'Query', 'decibel' ),
					),

					array(
						'type'        => 'text',
						'label'       => esc_html__( 'Tags', 'decibel' ),
						'param_name'  => 'tag',
						'description' => esc_html__( 'Include only one or several tags. Paste tag slug(s) separated by a comma', 'decibel' ),
						'placeholder' => esc_html__( 'my-tag, other-tag', 'decibel' ),
						'group'       => esc_html__( 'Query', 'decibel' ),
					),

					array(
						'type'        => 'text',
						'label'       => esc_html__( 'Exclude Tags by ID', 'decibel' ),
						'param_name'  => 'tag_exclude',
						'description' => esc_html__( 'Exclude only one or several tags. Paste tag ID(s) separated by a comma', 'decibel' ),
						'placeholder' => esc_html__( '456, 756', 'decibel' ),
						'group'       => esc_html__( 'Query', 'decibel' ),
					),

					array(
						'type'        => 'select',
						'label'       => esc_html__( 'Order by', 'decibel' ),
						'param_name'  => 'orderby',
						'options'     => decibel_order_by_values(),
						'save_always' => true,
						'description' => sprintf( decibel_kses( __( 'Select how to sort retrieved posts. More at %s.', 'decibel' ) ), 'WordPress codex page' ),
						'group'       => esc_html__( 'Query', 'decibel' ),
					),

					array(
						'type'        => 'select',
						'label'       => esc_html__( 'Sort order', 'decibel' ),
						'param_name'  => 'order',
						'options'     => decibel_order_way_values(),
						'save_always' => true,
						'description' => sprintf( decibel_kses( __( 'Designates the ascending or descending order. More at %s.', 'decibel' ) ), 'WordPress codex page' ),
						'group'       => esc_html__( 'Query', 'decibel' ),
					),

					array(
						'type'        => 'text',
						'label'       => esc_html__( 'Post IDs', 'decibel' ),
						'description' => esc_html__( 'By default, your last posts will be displayed. You can choose the posts you want to display by entering a list of IDs separated by a comma.', 'decibel' ),
						'param_name'  => 'include_ids',
						'group'       => esc_html__( 'Query', 'decibel' ),
					),

					array(
						'type'        => 'text',
						'label'       => esc_html__( 'Exclude Post IDs', 'decibel' ),
						'description' => esc_html__( 'You can choose the posts you don\'t want to display by entering a list of IDs separated by a comma.', 'decibel' ),
						'param_name'  => 'exclude_ids',
						'group'       => esc_html__( 'Query', 'decibel' ),
					),

					array(
						'param_name'  => 'columns',
						'label'       => esc_html__( 'Columns', 'decibel' ),
						'type'        => 'select',
						'options'     => array(
							2 => esc_html__( 'Two', 'decibel' ),
							3 => esc_html__( 'Three', 'decibel' ),
							4 => esc_html__( 'Four', 'decibel' ),
							5 => esc_html__( 'Five', 'decibel' ),
							6 => esc_html__( 'Six', 'decibel' ),
							1 => esc_html__( 'One', 'decibel' ),
						),
						'default'     => 3,
						'admin_label' => true,
						'condition'   => array(
							'post_display' => array( 'grid', 'grid_classic', 'masonry', 'masonry_modern' ),
						),
					),

					array(
						'type'        => 'text',
						'label'       => esc_html__( 'Extra class name', 'decibel' ),
						'param_name'  => 'el_class',
						'description' => esc_html__( 'Style particular content element differently - add a class name and refer to it in custom CSS.', 'decibel' ),
						'group'       => esc_html__( 'Extra', 'decibel' ),
					),
				),
			),
		)
	);
}

/**
 * Release Index
 */
function decibel_release_index_params() {
	return apply_filters(
		'decibel_release_index_params',
		array(
			'properties' => array(
				'name'          => esc_html__( 'Releases', 'decibel' ),
				'description'   => esc_html__( 'Display your releases using the theme layouts', 'decibel' ),
				'vc_base'       => 'wolf_core_release_index',
				'el_base'       => 'release-index',
				'vc_category'   => esc_html__( 'Content', 'decibel' ),
				'el_categories' => array( 'post-modules' ),
				'icon'          => 'linea-arrows linea-arrows-squares',
				'weight'        => 999,
			),

			'params'     => array_merge(
				decibel_common_module_params(),
				array(
					array(
						'param_name'  => 'release_display',
						'label'       => esc_html__( 'Release Display', 'decibel' ),
						'type'        => 'select',
						'options'     => apply_filters(
							'decibel_release_display_options',
							array(
								'grid' => esc_html__( 'Grid', 'decibel' ),
							)
						),
						'description' => esc_html__( 'The metro layout is not visible in preview mode yet.', 'decibel' ),
						'default'     => 'grid',
						'admin_label' => true,
					),

					array(
						'param_name'  => 'release_metro_pattern',
						'label'       => esc_html__( 'Metro Pattern', 'decibel' ),
						'type'        => 'select',
						'options'     => decibel_get_metro_patterns(),
						'default'     => 'auto',
						'condition'   => array(
							'release_display' => array( 'metro' ),
						),
						'admin_label' => true,
					),

					array(
						'param_name'   => 'release_alternate_thumbnail_position',
						'label'        => esc_html__( 'Alternate thumbnail position', 'decibel' ),
						'type'         => 'checkbox',
						'label_on'     => esc_html__( 'Yes', 'decibel' ),
						'label_off'    => esc_html__( 'No', 'decibel' ),
						'return_value' => 'yes',
						'condition'    => array(
							'release_display' => array( 'lateral' ),
						),
					),

					array(
						'param_name'  => 'release_layout',
						'label'       => esc_html__( 'Layout', 'decibel' ),
						'type'        => 'select',
						'options'     => apply_filters(
							'decibel_release_layout_options',
							array(
								'standard' => esc_html__( 'Classic', 'decibel' ),
								'overlay'  => esc_html__( 'Overlay', 'decibel' ),
							)
						),
						'default'     => 'standard',
						'admin_label' => true,
						'condition'   => array(
							'release_display' => array( 'grid', 'metro', 'masonry' ),
						),
					),

					array(
						'param_name'  => 'release_module',
						'label'       => esc_html__( 'Module', 'decibel' ),
						'type'        => 'select',
						'options'     => array(
							'grid'     => esc_html__( 'Grid', 'decibel' ),
							'carousel' => esc_html__( 'Carousel', 'decibel' ),
						),
						'description' => esc_html__( 'The carousel is not visible in preview mode yet.', 'decibel' ),
						'default'     => 'grid',
						'admin_label' => true,
						'condition'   => array(
							'release_display' => array( 'grid', 'animated_cover' ),
						),
					),

					array(
						'param_name'  => 'release_custom_thumbnail_size',
						'label'       => esc_html__( 'Custom Thumbnail Size', 'decibel' ),
						'type'        => 'text',
						'admin_label' => true,
						'placeholder' => '450x450',
					),

					array(
						'param_name'  => 'grid_padding',
						'label'       => esc_html__( 'Padding', 'decibel' ),
						'type'        => 'select',
						'options'     => array(
							'yes' => esc_html__( 'Yes', 'decibel' ),
							'no'  => esc_html__( 'No', 'decibel' ),
						),
						'default'     => 'yes',
						'admin_label' => true,
						'condition'   => array(
							'release_layout' => array( 'standard', 'overlay', 'label' ),
						),
					),
				),
				decibel_overlay_module_params( 'release' ),
				array(
					array(
						'label'        => esc_html__( 'Category Filter', 'decibel' ),
						'param_name'   => 'release_category_filter',
						'description'  => esc_html__( 'The pagination will be disabled.', 'decibel' ),
						'type'         => 'checkbox',
						'label_on'     => esc_html__( 'Yes', 'decibel' ),
						'label_off'    => esc_html__( 'No', 'decibel' ),
						'return_value' => 'yes',
						'admin_label'  => true,
						'condition'    => array(
							'release_display' => array( 'grid', 'animated_cover' ),
						),
					),

					array(
						'label'        => esc_html__( 'Filter Text Alignement', 'decibel' ),
						'param_name'   => 'release_category_filter_text_alignment',
						'type'         => 'choose',
						'options'      => array(
							'left'   => array(
								'title' => esc_html__( 'Left', 'decibel' ),
								'icon'  => 'eicon-text-align-left',
							),
							'center' => array(
								'title' => esc_html__( 'Center', 'decibel' ),
								'icon'  => 'eicon-text-align-center',
							),
							'right'  => array(
								'title' => esc_html__( 'Right', 'decibel' ),
								'icon'  => 'eicon-text-align-right',
							),
						),
						'condition'    => array(
							'release_category_filter' => 'yes',
						),
						'selectors'    => array(
							'{{WRAPPER}} .category-filter-release ul' => 'text-align:{{VALUE}};',
						),
						'page_builder' => 'elementor',
					),

					array(
						'label'        => esc_html__( 'Filter Text Alignement', 'decibel' ),
						'param_name'   => 'release_category_filter_text_alignment',
						'type'         => 'select',
						'options'      => array(
							'center' => esc_html__( 'Center', 'decibel' ),
							'left'   => esc_html__( 'Left', 'decibel' ),
							'right'  => esc_html__( 'Right', 'decibel' ),
						),
						'condition'    => array(
							'release_category_filter' => 'yes',
						),
						'page_builder' => 'vc',
					),

					array(
						'param_name'  => 'pagination',
						'label'       => esc_html__( 'Pagination', 'decibel' ),
						'type'        => 'select',
						'options'     => array(
							'none'                => esc_html__( 'None', 'decibel' ),
							'load_more'           => esc_html__( 'Load More', 'decibel' ),
							'standard_pagination' => esc_html__( 'Numeric Pagination', 'decibel' ),
							'link_to_discography' => esc_html__( 'Link to Discography', 'decibel' ),
						),
						'condition'   => array(
							'release_category_filter' => '',
						),
						'default'     => 'none',
						'admin_label' => true,
					),

					array(
						'type'        => 'text',
						'label'       => esc_html__( 'Include Band', 'decibel' ),
						'param_name'  => 'band_include',
						'description' => esc_html__( 'Enter one or several bands. Paste band slug(s) separated by a comma', 'decibel' ),
						'placeholder' => esc_html__( 'my-band, other-band', 'decibel' ),
						'group'       => esc_html__( 'Query', 'decibel' ),
					),

					array(
						'type'        => 'text',
						'label'       => esc_html__( 'Exclude Band', 'decibel' ),
						'param_name'  => 'band_exclude',
						'description' => esc_html__( 'Enter one or several bands. Paste band slug(s) separated by a comma', 'decibel' ),
						'placeholder' => esc_html__( 'my-band, other-band', 'decibel' ),
						'group'       => esc_html__( 'Query', 'decibel' ),
					),

					array(
						'type'        => 'text',
						'label'       => esc_html__( 'Include Type', 'decibel' ),
						'param_name'  => 'label_include',
						'description' => esc_html__( 'Enter one or several release types (from release tags). Paste type slug(s) separated by a comma', 'decibel' ),
						'placeholder' => esc_html__( 'my-type, other-type', 'decibel' ),
						'group'       => esc_html__( 'Query', 'decibel' ),
					),

					array(
						'type'        => 'text',
						'label'       => esc_html__( 'Exclude Type', 'decibel' ),
						'param_name'  => 'label_exclude',
						'description' => esc_html__( 'Enter one or several release types (from release tags). Paste type slug(s) separated by a comma', 'decibel' ),
						'placeholder' => esc_html__( 'my-type, other-type', 'decibel' ),
						'group'       => esc_html__( 'Query', 'decibel' ),
					),

					array(
						'type'        => 'select',
						'label'       => esc_html__( 'Order by', 'decibel' ),
						'param_name'  => 'orderby',
						'options'     => decibel_order_by_values(),
						'save_always' => true,
						'description' => sprintf( decibel_kses( __( 'Select how to sort retrieved posts. More at %s.', 'decibel' ) ), 'WordPress codex page' ),
						'group'       => esc_html__( 'Query', 'decibel' ),
					),

					array(
						'type'        => 'select',
						'label'       => esc_html__( 'Sort order', 'decibel' ),
						'param_name'  => 'order',
						'options'     => decibel_order_way_values(),
						'save_always' => true,
						'description' => sprintf( decibel_kses( __( 'Designates the ascending or descending order. More at %s.', 'decibel' ) ), 'WordPress codex page' ),
						'group'       => esc_html__( 'Query', 'decibel' ),
					),

					array(
						'type'        => 'text',
						'label'       => esc_html__( 'Post IDs', 'decibel' ),
						'description' => esc_html__( 'By default, your last posts will be displayed. You can choose the posts you want to display by entering a list of IDs separated by a comma.', 'decibel' ),
						'param_name'  => 'include_ids',
						'group'       => esc_html__( 'Query', 'decibel' ),
					),

					array(
						'type'        => 'text',
						'label'       => esc_html__( 'Exclude Post IDs', 'decibel' ),
						'description' => esc_html__( 'You can choose the posts you don\'t want to display by entering a list of IDs separated by a comma.', 'decibel' ),
						'param_name'  => 'exclude_ids',
						'group'       => esc_html__( 'Query', 'decibel' ),
					),

					array(
						'param_name'  => 'columns',
						'label'       => esc_html__( 'Columns', 'decibel' ),
						'type'        => 'select',
						'options'     => array(
							'default' => esc_html__( 'Auto', 'decibel' ),
							2         => esc_html__( 'Two', 'decibel' ),
							3         => esc_html__( 'Three', 'decibel' ),
							4         => esc_html__( 'Four', 'decibel' ),
							5         => esc_html__( 'Five', 'decibel' ),
							6         => esc_html__( 'Six', 'decibel' ),
							1         => esc_html__( 'One', 'decibel' ),
						),
						'default'     => 3,
						'admin_label' => true,
						'condition'   => array(
							'release_display' => array( 'grid', 'animated_cover' ),
						),
					),
				),
			),
		)
	);
}

/**
 * Work Index
 */
function decibel_work_index_params() {
	return apply_filters(
		'decibel_work_index_params',
		array(
			'properties' => array(
				'name'          => esc_html__( 'Works', 'decibel' ),
				'description'   => esc_html__( 'Display your works using the theme layouts', 'decibel' ),
				'vc_base'       => 'wolf_core_work_index',
				'el_base'       => 'work-index',
				'vc_category'   => esc_html__( 'Content', 'decibel' ),
				'el_categories' => array( 'post-modules' ),
				'icon'          => 'linea-arrows linea-arrows-squares',
				'weight'        => 999,
				'scripts'       => array( 'inview' ),
			),

			'params'     => array_merge(
				decibel_common_module_params(
					apply_filters( 'decibel_work_animation_condition', array( 'work_display' => array( 'metro', 'grid', 'masonry' ) ) ),
					apply_filters( 'decibel_work_post_count_condition', array( 'work_display' => array( 'metro', 'grid', 'masonry' ) ) )
				),
				array(
					array(
						'param_name'  => 'work_display',
						'label'       => esc_html__( 'Work Display', 'decibel' ),
						'type'        => 'select',
						'options'     => apply_filters(
							'decibel_work_display_options',
							array(
								'grid' => esc_html__( 'Grid', 'decibel' ),
							)
						),
						'description' => esc_html__( 'The metro layout is not visible in preview mode yet.', 'decibel' ),
						'default'     => 'grid',
						'admin_label' => true,
					),

					array(
						'param_name'  => 'work_metro_pattern',
						'label'       => esc_html__( 'Metro Pattern', 'decibel' ),
						'type'        => 'select',
						'options'     => decibel_get_metro_patterns(),
						'default'     => 'auto',
						'condition'   => array(
							'work_display' => array( 'metro' ),
						),
						'admin_label' => true,
					),

					array(
						'param_name'   => 'work_alternate_thumbnail_position',
						'label'        => esc_html__( 'Alternate thumbnail position', 'decibel' ),
						'type'         => 'checkbox',
						'label_on'     => esc_html__( 'Yes', 'decibel' ),
						'label_off'    => esc_html__( 'No', 'decibel' ),
						'return_value' => 'yes',
						'condition'    => array(
							'work_display' => array( 'lateral' ),
						),
					),

					array(
						'param_name'  => 'work_layout',
						'label'       => esc_html__( 'Layout', 'decibel' ),
						'type'        => 'select',
						'options'     => apply_filters(
							'decibel_work_layout_options',
							array(
								'standard' => esc_html__( 'Classic', 'decibel' ),
								'overlay'  => esc_html__( 'Overlay', 'decibel' ),
							)
						),
						'default'     => 'standard',
						'admin_label' => true,
						'condition'   => array(
							'work_display' => array( 'grid', 'metro', 'masonry' ),
						),
					),

					array(
						'param_name'  => 'work_module',
						'label'       => esc_html__( 'Module', 'decibel' ),
						'type'        => 'select',
						'options'     => array(
							'grid'     => esc_html__( 'Grid', 'decibel' ),
							'carousel' => esc_html__( 'Carousel', 'decibel' ),
						),
						'description' => esc_html__( 'The carousel is not visible in preview mode yet.', 'decibel' ),
						'default'     => 'grid',
						'admin_label' => true,
						'condition'   => array(
							'work_display' => array( 'grid' ),
						),
					),

					array(
						'param_name'  => 'work_thumbnail_size',
						'label'       => esc_html__( 'Thumbnail Size', 'decibel' ),
						'type'        => 'select',
						'options'     => array(
							'standard'  => esc_html__( 'Default Thumbnail', 'decibel' ),
							'landscape' => esc_html__( 'Landscape', 'decibel' ),
							'square'    => esc_html__( 'Square', 'decibel' ),
							'portrait'  => esc_html__( 'Portrait', 'decibel' ),
							'custom'    => esc_html__( 'Custom', 'decibel' ),
						),
						'default'     => 'standard',
						'admin_label' => true,
						'condition'   => array(
							'work_display' => array( 'grid' ),
						),
					),

					array(
						'param_name'  => 'work_custom_thumbnail_size',
						'label'       => esc_html__( 'Custom Thumbnail Size', 'decibel' ),
						'type'        => 'text',
						'admin_label' => true,
						'placeholder' => '450x450',
						'condition'   => array(
							'work_thumbnail_size' => array( 'custom' ),
						),
					),

					array(
						'param_name'  => 'grid_padding',
						'label'       => esc_html__( 'Padding', 'decibel' ),
						'type'        => 'select',
						'options'     => array(
							'yes' => esc_html__( 'Yes', 'decibel' ),
							'no'  => esc_html__( 'No', 'decibel' ),
						),
						'default'     => 'yes',
						'admin_label' => true,
						'condition'   => array(
							'work_display' => array( 'grid', 'metro', 'masonry' ),
						),
					),
					/*
					array(
						'param_name' => 'grid_padding_value',
						'label'      => esc_html__( 'Padding', 'decibel' ),
						'type'       => 'slider',
						'default'    => array(
							'unit' => array( 'px', '%' ),
							'size' => '20',
						),
						'selectors'   => array(
							'{{WRAPPER}} .entry-work' => 'padding:{{VALUE}};',
						),
						'size_units' => 'px',
						'condition'  => array(
							'grid_padding' => array( 'yes' ),
						),
					),
					*/
				),
				decibel_overlay_module_params( 'work' ),
				array(
					array(
						'label'        => esc_html__( 'Category Filter', 'decibel' ),
						'param_name'   => 'work_category_filter',
						'description'  => esc_html__( 'The pagination will be disabled.', 'decibel' ),
						'type'         => 'checkbox',
						'label_on'     => esc_html__( 'Yes', 'decibel' ),
						'label_off'    => esc_html__( 'No', 'decibel' ),
						'return_value' => 'yes',
						'admin_label'  => true,
						'condition'    => array(
							'work_display' => array( 'grid', 'masonry' ),
						),
					),

					array(
						'label'        => esc_html__( 'Filter Text Alignement', 'decibel' ),
						'param_name'   => 'work_category_filter_text_alignment',
						'type'         => 'choose',
						'options'      => array(
							'left'   => array(
								'title' => esc_html__( 'Left', 'decibel' ),
								'icon'  => 'eicon-text-align-left',
							),
							'center' => array(
								'title' => esc_html__( 'Center', 'decibel' ),
								'icon'  => 'eicon-text-align-center',
							),
							'right'  => array(
								'title' => esc_html__( 'Right', 'decibel' ),
								'icon'  => 'eicon-text-align-right',
							),
						),
						'condition'    => array(
							'work_category_filter' => 'yes',
						),
						'selectors'    => array(
							'{{WRAPPER}} .category-filter-work ul' => 'text-align:{{VALUE}};',
						),
						'page_builder' => 'elementor',
					),

					array(
						'label'        => esc_html__( 'Filter Text Alignement', 'decibel' ),
						'param_name'   => 'work_category_filter_text_alignment',
						'type'         => 'select',
						'options'      => array(
							'center' => esc_html__( 'Center', 'decibel' ),
							'left'   => esc_html__( 'Left', 'decibel' ),
							'right'  => esc_html__( 'Right', 'decibel' ),
						),
						'condition'    => array(
							'work_category_filter' => 'yes',
						),
						'page_builder' => 'vc',
					),

					array(
						'param_name'  => 'pagination',
						'label'       => esc_html__( 'Pagination', 'decibel' ),
						'type'        => 'select',
						'options'     => array(
							'none'                => esc_html__( 'None', 'decibel' ),
							'load_more'           => esc_html__( 'Load More', 'decibel' ),
							'standard_pagination' => esc_html__( 'Numeric Pagination', 'decibel' ),
							'link_to_portfolio'   => esc_html__( 'Link to Portfolio', 'decibel' ),
						),
						'condition'   => array(
							'work_category_filter' => '',
							'work_display'         => array( 'grid', 'masonry', 'metro' ),
						),
						'default'     => 'none',
						'admin_label' => true,
					),

					array(
						'type'        => 'text',
						'label'       => esc_html__( 'Include Category', 'decibel' ),
						'param_name'  => 'work_type_include',
						'description' => esc_html__( 'Enter one or several work categories (from work tags). Paste category slug(s) separated by a comma', 'decibel' ),
						'placeholder' => esc_html__( 'my-category, other-category', 'decibel' ),
						'group'       => esc_html__( 'Query', 'decibel' ),
					),

					array(
						'category'    => 'text',
						'label'       => esc_html__( 'Exclude Category', 'decibel' ),
						'param_name'  => 'work_type_exclude',
						'description' => esc_html__( 'Enter one or several work categories (from work tags). Paste category slug(s) separated by a comma', 'decibel' ),
						'placeholder' => esc_html__( 'my-category, other-category', 'decibel' ),
						'group'       => esc_html__( 'Query', 'decibel' ),
					),

					array(
						'type'        => 'text',
						'label'       => esc_html__( 'Include Format', 'decibel' ),
						'param_name'  => 'work_post_format_include',
						'description' => esc_html__( 'Enter one or several post format. Paste slug(s) separated by a comma', 'decibel' ),
						'placeholder' => esc_html__( 'gallery, video', 'decibel' ),
						'group'       => esc_html__( 'Query', 'decibel' ),
					),

					array(
						'category'    => 'text',
						'label'       => esc_html__( 'Exclude Format', 'decibel' ),
						'param_name'  => 'work_post_format_exclude',
						'description' => esc_html__( 'Enter one or several post format. Paste slug(s) separated by a comma', 'decibel' ),
						'placeholder' => esc_html__( 'gallery, video', 'decibel' ),
						'group'       => esc_html__( 'Query', 'decibel' ),
					),

					array(
						'type'        => 'select',
						'label'       => esc_html__( 'Order by', 'decibel' ),
						'param_name'  => 'orderby',
						'options'     => decibel_order_by_values(),
						'save_always' => true,
						'description' => sprintf( decibel_kses( __( 'Select how to sort retrieved posts. More at %s.', 'decibel' ) ), 'WordPress codex page' ),
						'group'       => esc_html__( 'Query', 'decibel' ),
					),

					array(
						'type'        => 'select',
						'label'       => esc_html__( 'Sort order', 'decibel' ),
						'param_name'  => 'order',
						'options'     => decibel_order_way_values(),
						'save_always' => true,
						'description' => sprintf( decibel_kses( __( 'Designates the ascending or descending order. More at %s.', 'decibel' ) ), 'WordPress codex page' ),
						'group'       => esc_html__( 'Query', 'decibel' ),
					),

					array(
						'type'        => 'text',
						'label'       => esc_html__( 'Post IDs', 'decibel' ),
						'description' => esc_html__( 'By default, your last posts will be displayed. You can choose the posts you want to display by entering a list of IDs separated by a comma.', 'decibel' ),
						'param_name'  => 'include_ids',
						'group'       => esc_html__( 'Query', 'decibel' ),
					),

					array(
						'type'        => 'text',
						'label'       => esc_html__( 'Exclude Post IDs', 'decibel' ),
						'description' => esc_html__( 'You can choose the posts you don\'t want to display by entering a list of IDs separated by a comma.', 'decibel' ),
						'param_name'  => 'exclude_ids',
						'group'       => esc_html__( 'Query', 'decibel' ),
					),

					array(
						'param_name'  => 'columns',
						'label'       => esc_html__( 'Columns', 'decibel' ),
						'type'        => 'select',
						'options'     => array(
							'default' => esc_html__( 'Auto', 'decibel' ),
							2         => esc_html__( 'Two', 'decibel' ),
							3         => esc_html__( 'Three', 'decibel' ),
							4         => esc_html__( 'Four', 'decibel' ),
							5         => esc_html__( 'Five', 'decibel' ),
							6         => esc_html__( 'Six', 'decibel' ),
							1         => esc_html__( 'One', 'decibel' ),
						),
						'default'     => 3,
						'admin_label' => true,
						'condition'   => array(
							'work_display' => array( 'grid', 'masonry' ),
						),
					),
				),
			),
		)
	);
}


/**
 * Product Index
 */
function decibel_product_index_params() {
	return apply_filters(
		'decibel_product_index_params',
		array(
			'properties' => array(
				'name'          => esc_html__( 'Products', 'decibel' ),
				'description'   => esc_html__( 'Display your products using the theme layouts', 'decibel' ),
				'vc_base'       => 'wolf_core_product_index',
				'el_base'       => 'product-index',
				'vc_category'   => esc_html__( 'Content', 'decibel' ),
				'el_categories' => array( 'post-modules' ),
				'icon'          => 'linea-arrows linea-arrows-squares',
				'weight'        => 999,
			),

			'params'     => array_merge(
				decibel_common_module_params(),
				array(
					array(
						'param_name'  => 'product_display',
						'label'       => esc_html__( 'Post Display', 'decibel' ),
						'type'        => 'select',
						'options'     => apply_filters(
							'decibel_product_display_options',
							array(
								'grid' => esc_html__( 'Grid', 'decibel' ),
							)
						),
						'default'     => 'grid',
						'admin_label' => true,
					),

					array(
						'param_name'  => 'product_metro_pattern',
						'label'       => esc_html__( 'Metro Pattern', 'decibel' ),
						'type'        => 'select',
						'options'     => decibel_get_metro_patterns(),
						'default'     => 'auto',
						'condition'   => array(
							'product_display' => array( 'metro' ),
						),
						'admin_label' => true,
					),

					/*
					array(
						'label'        => esc_html__( 'Product Text Alignement', 'decibel' ),
						'param_name'   => 'product_text_align',
						'type'         => 'choose',
						'options'      => array(
							'left'   => array(
								'title' => esc_html__( 'Left', 'decibel' ),
								'icon'  => 'eicon-text-align-left',
							),
							'center' => array(
								'title' => esc_html__( 'Center', 'decibel' ),
								'icon'  => 'eicon-text-align-center',
							),
							'right'  => array(
								'title' => esc_html__( 'Right', 'decibel' ),
								'icon'  => 'eicon-text-align-right',
							),
						),
						'selectors'    => array(
							'{{WRAPPER}} .entry-product' => 'margin-{{VALUE}}: 0;',
						),
						'page_builder' => 'elementor',
					),*/

					/*
					array(
						'type'         => 'select',
						'label'        => esc_html__( 'Product Text Alignement', 'decibel' ),
						'param_name'   => 'product_text_align',
						'options'      => array(
							'center' => esc_html__( 'Center', 'decibel' ),
							'left'   => esc_html__( 'Left', 'decibel' ),
							'right'  => esc_html__( 'Right', 'decibel' ),
						),
						'page_builder' => 'vc',
					),*/

					array(
						'param_name'  => 'product_meta',
						'label'       => esc_html__( 'Type', 'decibel' ),
						'type'        => 'select',
						'options'     => array(
							'all'          => esc_html__( 'All', 'decibel' ),
							'featured'     => esc_html__( 'Featured', 'decibel' ),
							'onsale'       => esc_html__( 'On Sale', 'decibel' ),
							'best_selling' => esc_html__( 'Best Selling', 'decibel' ),
							'top_rated'    => esc_html__( 'Top Rated', 'decibel' ),
						),
						'admin_label' => true,
					),

					array(
						'param_name'  => 'product_module',
						'label'       => esc_html__( 'Module', 'decibel' ),
						'type'        => 'select',
						'options'     => array(
							'grid'     => esc_html__( 'Grid', 'decibel' ),
							'carousel' => esc_html__( 'Carousel', 'decibel' ),
						),
						'default'     => 'grid',
						'admin_label' => true,
						'condition'   => array(
							'product_display' => array( 'grid', 'grid_classic', 'grid_modern' ),
						),
					),

					array(
						'param_name'  => 'columns',
						'label'       => esc_html__( 'Columns', 'decibel' ),
						'type'        => 'select',
						'options'     => array(
							'default' => esc_html__( 'Auto', 'decibel' ),
							2         => esc_html__( 'Two', 'decibel' ),
							3         => esc_html__( 'Three', 'decibel' ),
							4         => esc_html__( 'Four', 'decibel' ),
							5         => esc_html__( 'Five', 'decibel' ),
							6         => esc_html__( 'Six', 'decibel' ),
							1         => esc_html__( 'One', 'decibel' ),
						),
						'default'     => 3,
						'admin_label' => true,
						'condition'   => array(
							'product_display' => array( 'grid' ),
						),
					),

					array(
						'type'        => 'text',
						'label'       => esc_html__( 'Category', 'decibel' ),
						'param_name'  => 'product_cat',
						'description' => esc_html__( 'Include only one or several categories. Paste category slug(s) separated by a comma', 'decibel' ),
						'placeholder' => esc_html__( 'my-category, other-category', 'decibel' ),
						'group'       => esc_html__( 'Query', 'decibel' ),
					),

					array(
						'param_name'  => 'grid_padding',
						'label'       => esc_html__( 'Padding', 'decibel' ),
						'type'        => 'select',
						'options'     => array(
							'yes' => esc_html__( 'Yes', 'decibel' ),
							'no'  => esc_html__( 'No', 'decibel' ),
						),
						'default'     => 'yes',
						'admin_label' => true,
						'condition'   => array(
							'product_display' => array( 'grid', 'masonry', 'metro' ),
						),
					),

					array(
						'param_name'  => 'pagination',
						'label'       => esc_html__( 'Pagination', 'decibel' ),
						'type'        => 'select',
						'options'     => array(
							'none'                  => esc_html__( 'None', 'decibel' ),
							'load_more'             => esc_html__( 'Load More', 'decibel' ),
							'standard_pagination'   => esc_html__( 'Numeric Pagination', 'decibel' ),
							'link_to_shop_category' => esc_html__( 'Link to Category', 'decibel' ),
							'link_to_shop'          => esc_html__( 'Link to Shop Archive', 'decibel' ),
						),
						'default'     => 'none',
						'admin_label' => true,
					),

					array(
						'param_name'  => 'product_category_link_id',
						'label'       => esc_html__( 'Category Link', 'decibel' ),
						'type'        => 'select',
						'options'     => decibel_get_product_cat_dropdown_options(),
						'condition'   => array(
							'pagination' => array( 'link_to_shop_category' ),
						),
						'admin_label' => true,
					),
				),
			),
		)
	);
}

/**
 * Artist Index
 */
function decibel_artist_index_params() {
	return apply_filters(
		'decibel_artist_index_params',
		array(
			'properties' => array(
				'name'          => esc_html__( 'Artists', 'decibel' ),
				'description'   => esc_html__( 'Display your artists using the theme layouts', 'decibel' ),
				'vc_base'       => 'wolf_core_artist_index',
				'el_base'       => 'artist-index',
				'vc_category'   => esc_html__( 'Content', 'decibel' ),
				'el_categories' => array( 'post-modules' ),
				'icon'          => 'linea-arrows linea-arrows-squares',
				'weight'        => 999,
			),

			'params'     => array_merge(
				decibel_common_module_params(
					apply_filters( 'decibel_artist_animation_condition', array( 'artist_display' => array( 'metro', 'grid', 'masonry' ) ) ),
					apply_filters( 'decibel_artist_post_count_condition', array( 'artist_display' => array( 'metro', 'grid', 'masonry' ) ) )
				),
				array(

					array(
						'param_name'  => 'artist_display',
						'label'       => esc_html__( 'Artist Display', 'decibel' ),
						'type'        => 'select',
						'options'     => apply_filters(
							'decibel_artist_display_options',
							array(
								'list' => esc_html__( 'List', 'decibel' ),
							)
						),
						'admin_label' => true,
					),

					array(
						'param_name'  => 'artist_metro_pattern',
						'label'       => esc_html__( 'Metro Pattern', 'decibel' ),
						'type'        => 'select',
						'options'     => decibel_get_metro_patterns(),
						'default'     => 'auto',
						'condition'   => array(
							'artist_display' => 'metro',
						),
						'admin_label' => true,
					),

					array(
						'param_name'  => 'artist_module',
						'label'       => esc_html__( 'Module', 'decibel' ),
						'type'        => 'select',
						'options'     => array(
							'grid'     => esc_html__( 'Grid', 'decibel' ),
							'carousel' => esc_html__( 'Carousel', 'decibel' ),
						),
						'description' => esc_html__( 'The carousel is not visible in preview mode yet.', 'decibel' ),
						'default'     => 'grid',
						'admin_label' => true,
						'condition'   => array(
							'artist_display' => array( 'grid' ),
						),
					),

					array(
						'param_name'  => 'artist_thumbnail_size',
						'label'       => esc_html__( 'Thumbnail Size', 'decibel' ),
						'type'        => 'select',
						'options'     => array(
							'standard'  => esc_html__( 'Default Thumbnail', 'decibel' ),
							'landscape' => esc_html__( 'Landscape', 'decibel' ),
							'square'    => esc_html__( 'Square', 'decibel' ),
							'portrait'  => esc_html__( 'Portrait', 'decibel' ),
							'custom'    => esc_html__( 'Custom', 'decibel' ),
						),
						'admin_label' => true,
						'condition'   => array(
							'artist_display' => array( 'grid', 'offgrid' ),
						),
					),

					array(
						'param_name'  => 'artist_custom_thumbnail_size',
						'label'       => esc_html__( 'Custom Thumbnail Size', 'decibel' ),
						'type'        => 'text',
						'admin_label' => true,
						'placeholder' => '415x230',
						'condition'   => array(
							'artist_thumbnail_size' => 'custom',
						),
					),

					array(
						'param_name'  => 'artist_layout',
						'label'       => esc_html__( 'Layout', 'decibel' ),
						'type'        => 'select',
						'options'     => array(
							'standard' => esc_html__( 'Classic', 'decibel' ),
							'overlay'  => esc_html__( 'Overlay', 'decibel' ),
						),
						'admin_label' => true,
						'condition'   => array(
							'artist_display' => array( 'grid', 'masonry', 'offgrid', 'metro' ),
						),
						'description' => esc_html__( 'The metro layout is not visible in preview mode yet.', 'decibel' ),
					),

					array(
						'param_name'  => 'grid_padding',
						'label'       => esc_html__( 'Padding', 'decibel' ),
						'type'        => 'select',
						'options'     => array(
							'yes' => esc_html__( 'Yes', 'decibel' ),
							'no'  => esc_html__( 'No', 'decibel' ),
						),
						'default'     => 'yes',
						'admin_label' => true,
						'condition'   => array(
							'artist_display!' => array( 'interactive-link' ),
						),
					),

					array(
						'label'      => esc_html__( 'Caption Text Alignement', 'decibel' ),
						'param_name' => 'caption_text_alignment',
						'type'       => 'select',
						'options'    => array(
							esc_html__( 'Center', 'decibel' ) => 'center',
							esc_html__( 'Left', 'decibel' ) => 'left',
							esc_html__( 'Right', 'decibel' ) => 'right',
						),
						'condition'  => array(
							'element'            => 'artist_display',
							'value_not_equal_to' => array( 'list_minimal' ),
						),
					),

					array(
						'label'      => esc_html__( 'Caption Vertical Alignement', 'decibel' ),
						'param_name' => 'caption_v_align',
						'type'       => 'select',
						'options'    => array(
							esc_html__( 'Middle', 'decibel' ) => 'middle',
							esc_html__( 'Bottom', 'decibel' ) => 'bottom',
							esc_html__( 'Top', 'decibel' ) => 'top',
						),
						'condition'  => array(
							'element'            => 'artist_display',
							'value_not_equal_to' => array( 'list_minimal' ),
						),
					),

				),
				decibel_overlay_module_params( 'artist' ),
				array(
					array(
						'label'        => esc_html__( 'Category Filter', 'decibel' ),
						'param_name'   => 'artist_category_filter',
						'type'         => 'checkbox',
						'label_on'     => esc_html__( 'Yes', 'decibel' ),
						'label_off'    => esc_html__( 'No', 'decibel' ),
						'return_value' => 'no',
						'admin_label'  => true,
						'description'  => esc_html__( 'The pagination will be disabled.', 'decibel' ),
						'condition'   => array(
							'artist_display!' => array( 'interactive-link' ),
						),
					),

					array(
						'label'        => esc_html__( 'Filter Text Alignement', 'decibel' ),
						'param_name'   => 'artist_category_filter_text_alignment',
						'type'         => 'choose',
						'options'      => array(
							'left'   => array(
								'title' => esc_html__( 'Left', 'decibel' ),
								'icon'  => 'eicon-text-align-left',
							),
							'center' => array(
								'title' => esc_html__( 'Center', 'decibel' ),
								'icon'  => 'eicon-text-align-center',
							),
							'right'  => array(
								'title' => esc_html__( 'Right', 'decibel' ),
								'icon'  => 'eicon-text-align-right',
							),
						),
						'selectors'    => array(
							'{{WRAPPER}} .category-filter-artist ul' => 'text-align:{{VALUE}};',
						),
						'condition'    => array(
							'artist_category_filter' => 'yes',
						),
						'page_builder' => 'elementor',
					),

					array(
						'label'        => esc_html__( 'Filter Text Alignement', 'decibel' ),
						'param_name'   => 'artist_category_filter_text_alignment',
						'type'         => 'select',
						'options'      => array(
							'center' => esc_html__( 'Center', 'decibel' ),
							'left'   => esc_html__( 'Left', 'decibel' ),
							'right'  => esc_html__( 'Right', 'decibel' ),
						),
						'condition'    => array(
							'artist_category_filter' => 'yes',
						),
						'page_builder' => 'vc',
					),

					array(
						'param_name'  => 'pagination',
						'label'       => esc_html__( 'Pagination', 'decibel' ),
						'type'        => 'select',
						'options'     => array(
							'none'                => esc_html__( 'None', 'decibel' ),
							'load_more'           => esc_html__( 'Load More', 'decibel' ),
							'standard_pagination' => esc_html__( 'Numeric Pagination', 'decibel' ),
							'link_to_artists'     => esc_html__( 'Link to Archives', 'decibel' ),
						),
						'condition'   => array(
							'artist_category_filter' => '',
							'artist_display!' => array( 'interactive-link' ),
						),
						'admin_label' => true,
					),

					array(
						'type'        => 'text',
						'label'       => esc_html__( 'Include Category', 'decibel' ),
						'param_name'  => 'artist_genre_include',
						'description' => esc_html__( 'Enter one or several categories. Paste category slug(s) separated by a comma', 'decibel' ),
						'placeholder' => esc_html__( 'my-category, other-category', 'decibel' ),
						'group'       => esc_html__( 'Query', 'decibel' ),
					),

					array(
						'type'        => 'text',
						'label'       => esc_html__( 'Exclude Category', 'decibel' ),
						'param_name'  => 'artist_genre_exclude',
						'description' => esc_html__( 'Enter one or several categories. Paste category slug(s) separated by a comma', 'decibel' ),
						'placeholder' => esc_html__( 'my-category, other-category', 'decibel' ),
						'group'       => esc_html__( 'Query', 'decibel' ),
					),

					array(
						'type'        => 'select',
						'label'       => esc_html__( 'Order by', 'decibel' ),
						'param_name'  => 'orderby',
						'options'     => decibel_order_by_values(),
						'save_always' => true,
						'description' => sprintf( decibel_kses( __( 'Select how to sort retrieved posts. More at %s.', 'decibel' ) ), 'WordPress codex page' ),
						'group'       => esc_html__( 'Query', 'decibel' ),
					),

					array(
						'type'        => 'select',
						'label'       => esc_html__( 'Sort order', 'decibel' ),
						'param_name'  => 'order',
						'options'     => decibel_order_way_values(),
						'save_always' => true,
						'description' => sprintf( decibel_kses( __( 'Designates the ascending or descending order. More at %s.', 'decibel' ) ), 'WordPress codex page' ),
						'group'       => esc_html__( 'Query', 'decibel' ),
					),

					array(
						'type'        => 'text',
						'label'       => esc_html__( 'Post IDs', 'decibel' ),
						'description' => esc_html__( 'By default, your last posts will be displayed. You can choose the posts you want to display by entering a list of IDs separated by a comma.', 'decibel' ),
						'param_name'  => 'include_ids',
						'group'       => esc_html__( 'Query', 'decibel' ),
					),

					array(
						'type'        => 'text',
						'label'       => esc_html__( 'Exclude Post IDs', 'decibel' ),
						'description' => esc_html__( 'You can choose the posts you don\'t want to display by entering a list of IDs separated by a comma.', 'decibel' ),
						'param_name'  => 'exclude_ids',
						'group'       => esc_html__( 'Query', 'decibel' ),
					),

					array(
						'param_name'  => 'columns',
						'label'       => esc_html__( 'Columns', 'decibel' ),
						'type'        => 'select',
						'options'     => array(
							'default' => esc_html__( 'Auto', 'decibel' ),
							2         => esc_html__( 'Two', 'decibel' ),
							3         => esc_html__( 'Three', 'decibel' ),
							4         => esc_html__( 'Four', 'decibel' ),
							5         => esc_html__( 'Five', 'decibel' ),
							6         => esc_html__( 'Six', 'decibel' ),
							1         => esc_html__( 'One', 'decibel' ),
						),
						'default'     => 3,
						'admin_label' => true,
						'condition'   => array(
							'artist_display' => array( 'grid', 'masonry' ),
						),
					),
				),
			),
		)
	);
}

/**
 * Photo Album Index
 */
function decibel_album_index_params() {
	return apply_filters(
		'decibel_album_index_params',
		array(
			'properties' => array(
				'name'          => esc_html__( 'Photo Albums', 'decibel' ),
				'description'   => esc_html__( 'Display your albums using the theme layouts', 'decibel' ),
				'vc_base'       => 'wolf_core_album_index',
				'el_base'       => 'album-index',
				'vc_category'   => esc_html__( 'Content', 'decibel' ),
				'el_categories' => array( 'post-modules' ),
				'icon'          => 'linea-arrows linea-arrows-squares',
				'weight'        => 999,
			),

			'params'     => array_merge(
				decibel_common_module_params(),
				array(),
			),
		)
	);
}

/**
 * Video Index
 */
function decibel_video_index_params() {
	return apply_filters(
		'decibel_video_index_params',
		array(
			'properties' => array(
				'name'          => esc_html__( 'Videos', 'decibel' ),
				'description'   => esc_html__( 'Display your videos using the theme layouts', 'decibel' ),
				'vc_base'       => 'wolf_core_video_index',
				'el_base'       => 'video-index',
				'vc_category'   => esc_html__( 'Content', 'decibel' ),
				'el_categories' => array( 'post-modules' ),
				'icon'          => 'linea-arrows linea-arrows-squares',
				'weight'        => 999,
			),

			'params'     => array_merge(
				decibel_common_module_params(),
				array(
					array(
						'label'        => esc_html__( 'Show video on hover', 'decibel' ),
						'description'  => esc_html__( 'It is recommended to set upload a video sample mp4 file in your video post options below the text editor.', 'decibel' ),
						'param_name'   => 'video_preview',
						'type'         => 'checkbox',
						'label_on'     => esc_html__( 'Yes', 'decibel' ),
						'label_off'    => esc_html__( 'No', 'decibel' ),
						'return_value' => 'yes',
						'admin_label'  => true,
					),

					array(
						'param_name'  => 'video_module',
						'label'       => esc_html__( 'Module', 'decibel' ),
						'type'        => 'select',
						'options'     => array(
							'grid'     => esc_html__( 'Grid', 'decibel' ),
							'carousel' => esc_html__( 'Carousel', 'decibel' ),
						),
						'description' => esc_html__( 'The carousel is not visible in preview mode yet.', 'decibel' ),
						'default'     => 'grid',
						'admin_label' => true,
					),

					array(
						'param_name'  => 'video_custom_thumbnail_size',
						'label'       => esc_html__( 'Custom Thumbnail Size', 'decibel' ),
						'type'        => 'text',
						'admin_label' => true,
						'placeholder' => '415x230',
					),

					array(
						'param_name'  => 'grid_padding',
						'label'       => esc_html__( 'Padding', 'decibel' ),
						'type'        => 'select',
						'options'     => array(
							'yes' => esc_html__( 'Yes', 'decibel' ),
							'no'  => esc_html__( 'No', 'decibel' ),
						),
						'default'     => 'yes',
						'admin_label' => true,
					),

					array(
						'param_name'  => 'video_onclick',
						'label'       => esc_html__( 'On Click', 'decibel' ),
						'type'        => 'select',
						'options'     => array(
							'lightbox' => esc_html__( 'Open Video in Lightbox', 'decibel' ),
							'default'  => esc_html__( 'Go to the Video Page', 'decibel' ),
						),
						'default'     => 'lightbox',
						'admin_label' => true,
					),

					array(
						'label'        => esc_html__( 'Category Filter', 'decibel' ),
						'param_name'   => 'video_category_filter',
						'type'         => 'checkbox',
						'label_on'     => esc_html__( 'Yes', 'decibel' ),
						'label_off'    => esc_html__( 'No', 'decibel' ),
						'return_value' => 'yes',
						'admin_label'  => true,
						'description'  => esc_html__( 'The pagination will be disabled.', 'decibel' ),
					),

					array(
						'label'        => esc_html__( 'Filter Text Alignement', 'decibel' ),
						'param_name'   => 'video_category_filter_text_alignment',
						'type'         => 'choose',
						'options'      => array(
							'left'   => array(
								'title' => esc_html__( 'Left', 'decibel' ),
								'icon'  => 'eicon-text-align-left',
							),
							'center' => array(
								'title' => esc_html__( 'Center', 'decibel' ),
								'icon'  => 'eicon-text-align-center',
							),
							'right'  => array(
								'title' => esc_html__( 'Right', 'decibel' ),
								'icon'  => 'eicon-text-align-right',
							),
						),
						'selectors'    => array(
							'{{WRAPPER}} .category-filter-video ul' => 'text-align:{{VALUE}};',
						),
						'condition'    => array(
							'video_category_filter' => 'yes',
						),
						'page_builder' => 'elementor',
					),

					array(
						'label'        => esc_html__( 'Filter Text Alignement', 'decibel' ),
						'param_name'   => 'video_category_filter_text_alignment',
						'type'         => 'select',
						'options'      => array(
							'center' => esc_html__( 'Center', 'decibel' ),
							'left'   => esc_html__( 'Left', 'decibel' ),
							'right'  => esc_html__( 'Right', 'decibel' ),
						),
						'condition'    => array(
							'video_category_filter' => 'yes',
						),
						'page_builder' => 'vc',
					),

					array(
						'param_name'  => 'pagination',
						'label'       => esc_html__( 'Pagination', 'decibel' ),
						'type'        => 'select',
						'options'     => array(
							'none'                => esc_html__( 'None', 'decibel' ),
							'load_more'           => esc_html__( 'Load More', 'decibel' ),
							'standard_pagination' => esc_html__( 'Numeric Pagination', 'decibel' ),
							'link_to_videos'      => esc_html__( 'Link to Video Archives', 'decibel' ),
						),
						'condition'   => array(
							'video_category_filter' => '',
						),
						'default'     => 'none',
						'admin_label' => true,
					),

					array(
						'param_name'  => 'video_category_link_id',
						'label'       => esc_html__( 'Category', 'decibel' ),
						'type'        => 'select',
						'options'     => decibel_get_video_cat_dropdown_options(),
						'condition'   => array(
							'pagination' => 'link_to_video_category',
						),
						'admin_label' => true,
					),

					array(
						'type'        => 'text',
						'label'       => esc_html__( 'Include Category', 'decibel' ),
						'param_name'  => 'video_type_include',
						'description' => esc_html__( 'Enter one or several categories. Paste category slug(s) separated by a comma', 'decibel' ),
						'placeholder' => esc_html__( 'my-category, other-category', 'decibel' ),
						'group'       => esc_html__( 'Query', 'decibel' ),
					),

					array(
						'type'        => 'text',
						'label'       => esc_html__( 'Exclude Category', 'decibel' ),
						'param_name'  => 'video_type_exclude',
						'description' => esc_html__( 'Enter one or several categories. Paste category slug(s) separated by a comma', 'decibel' ),
						'placeholder' => esc_html__( 'my-category, other-category', 'decibel' ),
						'group'       => esc_html__( 'Query', 'decibel' ),
					),

					array(
						'type'        => 'text',
						'label'       => esc_html__( 'Include Tag', 'decibel' ),
						'param_name'  => 'video_tag_include',
						'description' => esc_html__( 'Enter one or several tags. Paste category slug(s) separated by a comma', 'decibel' ),
						'placeholder' => esc_html__( 'my-tag, other-tag', 'decibel' ),
						'group'       => esc_html__( 'Query', 'decibel' ),
					),

					array(
						'type'        => 'text',
						'label'       => esc_html__( 'Exclude Tag', 'decibel' ),
						'param_name'  => 'video_tag_exclude',
						'description' => esc_html__( 'Enter one or several tags. Paste category slug(s) separated by a comma', 'decibel' ),
						'placeholder' => esc_html__( 'my-tag, other-tag', 'decibel' ),
						'group'       => esc_html__( 'Query', 'decibel' ),
					),

					array(
						'type'        => 'select',
						'label'       => esc_html__( 'Order by', 'decibel' ),
						'param_name'  => 'orderby',
						'options'     => decibel_order_by_values(),
						'save_always' => true,
						'description' => sprintf( decibel_kses( __( 'Select how to sort retrieved posts. More at %s.', 'decibel' ) ), 'WordPress codex page' ),
						'group'       => esc_html__( 'Query', 'decibel' ),
					),

					array(
						'type'        => 'select',
						'label'       => esc_html__( 'Sort order', 'decibel' ),
						'param_name'  => 'order',
						'options'     => decibel_order_way_values(),
						'save_always' => true,
						'description' => sprintf( decibel_kses( __( 'Designates the ascending or descending order. More at %s.', 'decibel' ) ), 'WordPress codex page' ),
						'group'       => esc_html__( 'Query', 'decibel' ),
					),

					array(
						'type'        => 'text',
						'label'       => esc_html__( 'Post IDs', 'decibel' ),
						'description' => esc_html__( 'By default, your last posts will be displayed. You can choose the posts you want to display by entering a list of IDs separated by a comma.', 'decibel' ),
						'param_name'  => 'include_ids',
						'group'       => esc_html__( 'Query', 'decibel' ),
					),

					array(
						'type'        => 'text',
						'label'       => esc_html__( 'Exclude Post IDs', 'decibel' ),
						'description' => esc_html__( 'You can choose the posts you don\'t want to display by entering a list of IDs separated by a comma.', 'decibel' ),
						'param_name'  => 'exclude_ids',
						'group'       => esc_html__( 'Query', 'decibel' ),
					),

					array(
						'param_name'  => 'columns',
						'label'       => esc_html__( 'Columns', 'decibel' ),
						'type'        => 'select',
						'options'     => array(
							'default' => esc_html__( 'Auto', 'decibel' ),
							2         => esc_html__( 'Two', 'decibel' ),
							3         => esc_html__( 'Three', 'decibel' ),
							4         => esc_html__( 'Four', 'decibel' ),
							5         => esc_html__( 'Five', 'decibel' ),
							6         => esc_html__( 'Six', 'decibel' ),
							1         => esc_html__( 'One', 'decibel' ),
						),
						'default'     => 3,
						'admin_label' => true,
					),
				),
			),
		)
	);
}

/**
 * Event Index
 */
function decibel_event_index_params() {
	return apply_filters(
		'decibel_event_index_params',
		array(
			'properties' => array(
				'name'          => esc_html__( 'Events', 'decibel' ),
				'description'   => esc_html__( 'Display your events using the theme layouts', 'decibel' ),
				'vc_base'       => 'wolf_core_event_index',
				'el_base'       => 'event-index',
				'vc_category'   => esc_html__( 'Content', 'decibel' ),
				'el_categories' => array( 'post-modules' ),
				'icon'          => 'linea-arrows linea-arrows-squares',
				'weight'        => 999,
			),

			'params'     => array_merge(
				decibel_common_module_params(),
				array(
					array(
						'param_name'  => 'event_display',
						'label'       => esc_html__( 'Event Display', 'decibel' ),
						'type'        => 'select',
						'options'     => apply_filters(
							'decibel_event_display_options',
							array(
								'list' => esc_html__( 'List', 'decibel' ),
							)
						),
						'default'     => 'list',
						'admin_label' => true,
					),

					array(
						'param_name' => 'event_layout',
						'label'      => esc_html__( 'Event Layout', 'decibel' ),
						'type'       => 'hidden',
						'default'    => 'overlay',
					),

					array(
						'param_name'  => 'event_module',
						'label'       => esc_html__( 'Module', 'decibel' ),
						'type'        => 'select',
						'options'     => array(
							'grid'     => esc_html__( 'Grid', 'decibel' ),
							'carousel' => esc_html__( 'Carousel', 'decibel' ),
						),
						'description' => esc_html__( 'The carousel is not visible in preview mode yet.', 'decibel' ),
						'default'     => 'grid',
						'admin_label' => true,
						'condition'   => array(
							'event_display' => array( 'grid' ),
						),
					),

					array(
						'param_name'  => 'event_thumbnail_size',
						'label'       => esc_html__( 'Thumbnail Size', 'decibel' ),
						'type'        => 'select',
						'options'     => array(
							'standard'  => esc_html__( 'Default Thumbnail', 'decibel' ),
							'landscape' => esc_html__( 'Landscape', 'decibel' ),
							'square'    => esc_html__( 'Square', 'decibel' ),
							'portrait'  => esc_html__( 'Portrait', 'decibel' ),
							'custom'    => esc_html__( 'Custom', 'decibel' ),
						),
						'default'     => 'standard',
						'admin_label' => true,
						'condition'   => array(
							'event_display' => array( 'grid' ),
						),
					),

					array(
						'param_name'  => 'event_custom_thumbnail_size',
						'label'       => esc_html__( 'Custom Thumbnail Size', 'decibel' ),
						'type'        => 'text',
						'admin_label' => true,
						'placeholder' => '415x230',
						'condition'   => array(
							'event_thumbnail_size' => 'custom',
						),
					),
				),
				decibel_overlay_module_params( 'event' ),
				array(
					array(
						'param_name'  => 'grid_padding',
						'label'       => esc_html__( 'Padding', 'decibel' ),
						'type'        => 'select',
						'options'     => array(
							'yes' => esc_html__( 'Yes', 'decibel' ),
							'no'  => esc_html__( 'No', 'decibel' ),
						),
						'default'     => 'yes',
						'admin_label' => true,
						'condition'   => array(
							'event_display' => array( 'grid' ),
						),
					),

					array(
						'param_name'  => 'columns',
						'label'       => esc_html__( 'Columns', 'decibel' ),
						'type'        => 'select',
						'options'     => array(
							'default' => esc_html__( 'Auto', 'decibel' ),
							2         => esc_html__( 'Two', 'decibel' ),
							3         => esc_html__( 'Three', 'decibel' ),
							4         => esc_html__( 'Four', 'decibel' ),
							5         => esc_html__( 'Five', 'decibel' ),
							6         => esc_html__( 'Six', 'decibel' ),
							1         => esc_html__( 'One', 'decibel' ),
						),
						'default'     => 3,
						'admin_label' => true,
						'condition'   => array(
							'event_display' => array( 'grid' ),
						),
					),

					array(
						'param_name'  => 'event_location',
						'label'       => esc_html__( 'Location', 'decibel' ),
						'type'        => 'select',
						'options'     => array(
							'location' => esc_html__( 'Location', 'decibel' ),
							'venue'    => esc_html__( 'Venue', 'decibel' ),
						),
						'default'     => 'location',
						'admin_label' => true,
					),

					array(
						'param_name'  => 'pagination',
						'label'       => esc_html__( 'Pagination', 'decibel' ),
						'type'        => 'select',
						'options'     => array(
							'none'           => esc_html__( 'None', 'decibel' ),
							'link_to_events' => esc_html__( 'Link to Event Archives', 'decibel' ),
						),
						'default'     => 'link_to_events',
						'admin_label' => true,
					),

					array(
						'type'       => 'select',
						'label'      => esc_html__( 'Timeline', 'decibel' ),
						'param_name' => 'timeline',
						'options'    => array(
							'future' => esc_html__( 'Future', 'decibel' ),
							'past'   => esc_html__( 'Past', 'decibel' ),
						),
						'default'    => 'future',
						'group'      => esc_html__( 'Query', 'decibel' ),
					),

					array(
						'type'        => 'text',
						'label'       => esc_html__( 'Include Artist', 'decibel' ),
						'param_name'  => 'artist_include',
						'description' => esc_html__( 'Enter one or several bands. Paste category slug(s) separated by a comma', 'decibel' ),
						'placeholder' => esc_html__( 'my-category, other-category', 'decibel' ),
						'group'       => esc_html__( 'Query', 'decibel' ),
					),

					array(
						'type'        => 'text',
						'label'       => esc_html__( 'Exclude Artist', 'decibel' ),
						'param_name'  => 'artist_exclude',
						'description' => esc_html__( 'Enter one or several bands. Paste category slug(s) separated by a comma', 'decibel' ),
						'placeholder' => esc_html__( 'my-category, other-category', 'decibel' ),
						'group'       => esc_html__( 'Query', 'decibel' ),
					),
				)
			),
		)
	);
}

/**
 * Page Index
 */
function decibel_page_index_params() {
	return apply_filters(
		'decibel_page_index_params',
		array(
			'properties' => array(
				'name'          => esc_html__( 'Pages', 'decibel' ),
				'description'   => esc_html__( 'Display your pages using the theme layouts', 'decibel' ),
				'vc_base'       => 'wolf_core_page_index',
				'el_base'       => 'page-index',
				'vc_category'   => esc_html__( 'Content', 'decibel' ),
				'el_categories' => array( 'post-modules' ),
				'icon'          => 'linea-arrows linea-arrows-squares',
				'weight'        => 999,
			),

			'params'     => array_merge(
				decibel_common_module_params(),
				array()
			),
		)
	);
}
